document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('search-input');
    const searchButton = document.getElementById('search-button');
    const loading = document.getElementById('loading');
    const results = document.getElementById('results');
    const resultContent = document.getElementById('result-content');
    const sourcesList = document.getElementById('sources-list');

    // Function to handle the search
    async function performSearch() {
        const query = searchInput.value.trim();
        
        if (!query) {
            alert('Por favor, ingresa una pregunta o tema para investigar.');
            return;
        }
        
        // Show loading, hide results
        loading.classList.remove('hidden');
        results.classList.add('hidden');
        
        try {
            // Call the Pollinations.AI API
            const response = await fetch('https://text.pollinations.ai/openai-large', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    model: 'searchgpt',
                    messages: [
                        {
                            role: 'system',
                            content: 'Eres un asistente de investigación de IA avanzado diseñado para proporcionar información completa y precisa. Para cada consulta, realiza una investigación exhaustiva examinando múltiples fuentes de alta calidad. Verifica los hechos en diferentes referencias antes de presentarlos. Presenta puntos de vista equilibrados sobre temas controvertidos. Cita tus fuentes claramente y explica cualquier limitación en la información disponible. Prioriza la precisión sobre la velocidad y la profundidad sobre la brevedad. Tu objetivo es ofrecer la respuesta más confiable y bien investigada posible. Siempre debes incluir referencias confiables y fuentes consultadas.'
                        },
                        {
                            role: 'user',
                            content: query
                        }
                    ],
                    referrer: 'AIDeepResearcher'
                })
            });

            if (!response.ok) {
                throw new Error(`La solicitud a la API falló con estado ${response.status}`);
            }

            const data = await response.json();
            
            // Process and display the results
            displayResults(data, query);
            
        } catch (error) {
            console.error('Error al realizar la búsqueda:', error);
            resultContent.innerHTML = `
                <div class="error">
                    <p>Lo sentimos, hubo un error al procesar tu solicitud. Por favor, intenta de nuevo más tarde.</p>
                    <p>Detalles del error: ${error.message}</p>
                </div>
            `;
            
            // Hide loading, show results (with error)
            loading.classList.add('hidden');
            results.classList.remove('hidden');
        }
    }
    
    // Function to display search results
    function displayResults(data, query) {
        // Extract the response content
        const content = data.choices[0].message.content;
        
        // Parse content and extract sources
        let formattedContent = content;
        const sources = [];
        
        // Simple regex to find URLs in the content
        const urlRegex = /(https?:\/\/[^\s]+)/g;
        const urlMatches = content.match(urlRegex);
        
        if (urlMatches) {
            // Extract URLs as sources
            urlMatches.forEach(url => {
                const cleanUrl = url.replace(/[.,;:)]$/, ''); // Remove trailing punctuation
                if (!sources.includes(cleanUrl)) {
                    sources.push(cleanUrl);
                }
            });
            
            // Clean up the content by removing explicit URLs if needed
            formattedContent = content.replace(urlRegex, '');
        }
        
        // Display the main content
        resultContent.innerHTML = `<p>${formatContent(formattedContent)}</p>`;
        
        // Display sources
        sourcesList.innerHTML = '';
        if (sources.length > 0) {
            sources.forEach(source => {
                // Create a more user-friendly display of the URL
                let displayUrl = source;
                try {
                    const url = new URL(source);
                    displayUrl = url.hostname + url.pathname.substring(0, 20) + (url.pathname.length > 20 ? '...' : '');
                } catch (e) {
                    // If parsing fails, use the original URL
                }
                
                const li = document.createElement('li');
                li.innerHTML = `<a href="${source}" target="_blank" rel="noopener noreferrer">${displayUrl}</a>`;
                sourcesList.appendChild(li);
            });
        } else {
            // If no sources found, add a note
            const li = document.createElement('li');
            li.textContent = 'No se citaron fuentes específicas.';
            sourcesList.appendChild(li);
        }
        
        // Hide loading, show results
        loading.classList.add('hidden');
        results.classList.remove('hidden');
    }
    
    // Function to format content with paragraph breaks
    function formatContent(text) {
        // Replace newlines with paragraph breaks
        return text.replace(/\n\n/g, '</p><p>').replace(/\n/g, '<br>');
    }
    
    // Event listeners
    searchButton.addEventListener('click', performSearch);
    
    searchInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            performSearch();
        }
    });
});