import V3 from "./V3.js";
import {cos, sin} from "./MathUtils.js";

/**
 * Matrices 4x4.
 * Definición y operaciones de matrices de transformación 3D.
 * @author Melissa Méndez Servín.
 */
class Matrix4{
    identity(){
        return [ 1, 0, 0, 0,
                 0, 1, 0, 0,
                 0, 0, 1, 0,
                 0, 0, 0, 1 ];     
    }
    translate(tx, ty, tz){
        return [ 1, 0, 0, 0,
                 0, 1, 0, 0,
                 0, 0, 1, 0,
                 tx, ty, tz, 1 ];
    }
    scale(sx, sy, sz){
        return [ sx, 0, 0, 0,
                 0, sy, 0, 0,
                 0, 0, sz, 0,
                 0, 0, 0, 1 ];   
    }
    /**
     * Rotación sobre el eje Z
     * @param {*} theta ángulo en radianes
     */
    rotateZR(theta){
        let c = cos(theta);
        let s = sin(theta);
        return [    c, s, 0, 0, 
                    -s, c, 0, 0,
                    0, 0, 1, 0,
                    0, 0, 0, 1];
    }
    /**
     * Rotación sobre el eje X
     * @param {*} theta ángulo en radianes
     */
    rotateXR(theta){
        let c = cos(theta);
        let s = sin(theta);
        return  [   1,  0, 0, 0, 
                    0,  c, s, 0,
                    0, -s, c, 0,
                    0,  0, 0, 1];
    }
    /**
     * Rotación sobre el eje Y
     * @param {*} theta ángulo en radianes
     */
    rotateYR(theta){
        let c = cos(theta);
        let s = sin(theta);
        return [    c, 0, -s, 0, 
                    0, 1,  0, 0,
                    s, 0,  c, 0,
                    0, 0,  0, 1];
    }
    /**
     * Rotación sobre el eje Z
     * @param {*} theta ángulo en grados
     */
    rotateZ(theta){
        var thetaToRadians = this.degToRad(theta);
        let c = cos(thetaToRadians);
        let s = sin(thetaToRadians);
        return [  c, s, 0, 0, 
                 -s, c, 0, 0,
                  0, 0, 1, 0,
                  0, 0, 0, 1];
    }
    /**
     * Rotación sobre el eje X
     * @param {*} theta ángulo en grados
     */
    rotateX(theta){
        var thetaToRadians = this.degToRad(theta);
        let c = cos(thetaToRadians);
        let s = sin(thetaToRadians);
        return [ 1,  0, 0, 0, 
                 0,  c, s, 0,
                 0, -s, c, 0,
                 0,  0, 0, 1];
    }
    /**
     * Rotación sobre el eje Y
     * @param {*} theta ángulo en grados
     */
    rotateY(theta){
        var thetaToRadians = this.degToRad(theta);
        let c = cos(thetaToRadians);
        let s = sin(thetaToRadians);
        return [ c, 0, -s, 0, 
                 0, 1,  0, 0,
                 s, 0,  c, 0,
                 0, 0,  0, 1];
    }
    /**
     * 
     * [u.x, u.y, u.z, -pos dot u,
     *  v.x, v.y, v.z, -pos dot v,
     *  w.x, w.y, w.z, -pos dot w,
     *  0,   0,   0,   1 ]
     * 
     * @param {V3} pos 
     * @param {V3} target 
     * @param {V3} up 
     */
    camera(pos, target, up){
        var w = pos.sub(target).normalize();
        var u = up.cross(w).normalize();
        var v = w.cross(u);
        let change_of_basis = [ u.x, v.x, w.x, 0,
                                u.y, v.y, w.y, 0,
                                u.z, v.z, w.z, 0,
                                0, 0, 0, 1];
        return this.multiply(change_of_basis, this.translate(-pos.x, -pos.y, -pos.z));   
    }
    
    perspective(fovInDegrees, aspect, near, far){
        var fovInRadians = this.degToRad(fovInDegrees)
        let cot = 1 / Math.tan(fovInRadians/2);
        return  [ cot/aspect,  0,        0,          0,
                    0,         cot,        0,          0,
                    0,           0,    -(near + far)/(far - near), -1,
                    0,           0,    -(2*far*near)/(far - near),  0];
    }
    perspInf( l, r, t, b, n){
        return [       2*n/(r - l),               0,               0, 0,
                               0,       2*n/(t - b),               0, 0,
                               0,               0,               -1, -1,
                -(r + l)/(r - l), -(t + b)/(t - b),             -2*n, 0];
    }
    ortho( l, r, t, b, n, f){
        return [       2/(r - l),               0,               0, 0,
                               0,       2/(t - b),               0, 0,
                               0,               0,       2/(n - f), 0,
                -(r + l)/(r - l), -(t + b)/(t - b), -(f + n)/(n- f), 1];
    }
    frustrum( l, r, t, b, n, f){
        return [       2*n/(r - l),               0,               0, 0,
                               0,       2*n/(t - b),               0, 0,
                               0,               0,  -(f + n)/(f - n),-1,
                -(r + l)/(r - l), -(t + b)/(t - b), -(2*n*f)/(f - n), 0];
    }
    /**
     * Matriz que permite rotar un vector sobre un eje arbitrario usando la
     * fórmula de Rodrigues.
     * @param {V3} axis el eje arbitrario representado con un vector unitario.  
     * @param {*} theta el ángulo a rotar en radianes
     */
    rotate(axis, theta){
        let c = cos(theta);
        let s = sin(theta);
        let x = axis.x;
        let y = axis.y;
        let z = axis.z;
        return [ (1-c)* x*x + c,     (1-c)* x*y + (s*z), (1-c)* x*z - (s*y), 0,
                 (1-c)* x*y - (s*z), (1-c)* y*y + c,     (1-c)* y*z + (s*x), 0,
                 (1-c)* x*z + (s*y), (1-c)* y*z - (s*x), (1-c)* z*z + c,     0,
                 0, 0, 0, 1];
    }
    /**
     * A^T * B^T = (BA)^T, pues nos dan dos matrices transpuestas de entrada, 
     * es decir, A^T y B^T.
     * p_ij = (b_i0 * a_0j) + (b_i1 * a_1j) + (b_i2 * a_2j) + (b_i3 * a_3j) 
     */
    multiply(a,b){
        var p = [];
        for(var i= 0; i < 4; i++)
            for(var j= 0; j < 4; j++){
                p[4 * i + j] = 0;
                for( var k = 0; k <4; k++)
                     p[4 * i + j] += b[4* i + k] * a[4 * k + j];
            }
        return p;
    }
    /**
     * Multiplica de izq a der.
     * @param {*} matrices 
     * @returns 
     */
    multiplyArray(matrices){
        var finalMatrix = M4.identity();
        for(var i= 0; i < matrices.length; i++)
            finalMatrix = this.multiply(finalMatrix, matrices[i]);
        return finalMatrix;
    }
    transpose(m){
        return [ m[0], m[4], m[8], m[12],
                 m[1], m[5], m[9], m[13],
                 m[2], m[6], m[10], m[14],
                 m[3], m[7], m[11], m[15]
                ];
    }
    basisMatrix(e1, e2, e3){
        return [ e1.x, e1.y, e1.z, 0, 
                 e2.x, e2.y, e2.z, 0,
                 e3.x, e3.y, e3.z, 0,
                 0, 0, 0, 1];
    }
    /**
     * m = [m00, m01, m02, m03,
     *      m10, m11, m12, m13,
     *      m20, m21, m22, m23,
     *      m30, m31, m32, m33]
     * 
     *      m00, m01, m02,
     *      m10, m11, m12,
     *      m20, m21, m22
     * @param {*} 
     * @returns 
     */
    inverse(m) {
        let m00 = m[0] -1+1;
        let m01 = m[4] -1+1;
        let m02 = m[8] -1+1;
        let m03 = m[12]-1+1;
        let m10 = m[1] -1+1;
        let m11 = m[5] -1+1;
        let m12 = m[9] -1+1;
        let m13 = m[13]-1+1;
        let m20 = m[2] -1+1;
        let m21 = m[6] -1+1;
        let m22 = m[10]-1+1;
        let m23 = m[14]-1+1;
        let m30 = m[3] -1+1;
        let m31 = m[7] -1+1;
        let m32 = m[11]-1+1;
        let m33 = m[15]-1+1;

        let a00 =  (m11 * (m22 * m33 - m23 * m32) - m21 * (m12 * m33 - m13 * m32) + m31 * (m12 * m23 - m13 * m22));
        let a01 = -(m10 * (m22 * m33 - m23 * m32) - m20 * (m12 * m33 - m13 * m32) + m30 * (m12 * m23 - m13 * m22));
        let a02 =  (m10 * (m21 * m33 - m23 * m31) - m20 * (m11 * m33 - m13 * m31) + m30 * (m11 * m23 - m13 * m21));
        let a03 = -(m10 * (m21 * m32 - m22 * m31) - m20 * (m11 * m32 - m12 * m31) + m30 * (m11 * m22 - m12 * m21));
        
        let a10 = -(m01 * (m22 * m33 - m23 * m32) - m21 * (m02 * m33 - m03 * m32) + m31 * (m02 * m23 - m03 * m22));
        let a11 = (m00 * (m22 * m33 - m23 * m32) - m20 * (m02 * m33 - m03 * m32) + m30 * (m02 * m23 - m03 * m22));
        let a12 = -(m00 * (m21 * m33 - m23 * m31) - m20 * (m01 * m33 - m03 * m31) + m30 * (m01 * m23 - m03 * m21));
        let a13 = (m00 * (m21 * m32 - m22 * m31) - m20 * (m01 * m32 - m02 * m31) + m30 * (m01 * m22 - m02 * m21));
        
        let a20 = (m01 * (m12 * m33 - m13 * m32) - m11 * (m02 * m33 - m03 * m32) + m31 * (m02 * m13 - m03 * m12));
        let a21 = -(m00 * (m12 * m33 - m13 * m32) - m10 * (m02 * m33 - m03 * m32) + m30 * (m02 * m13 - m03 * m12));
        let a22 = (m00 * (m11 * m33 - m13 * m31) - m10 * (m01 * m33 - m03 * m31) + m30 * (m01 * m13 - m03 * m11));
        let a23 = -(m00 * (m11 * m32 - m12 * m31) - m10 * (m01 * m32 - m02 * m31) + m30 * (m01 * m12 - m02 * m11));
        
        let a30 = -(m01 * (m12 * m23 - m13 * m22) - m11 * (m02 * m23 - m03 * m22) + m21 * (m02 * m13 - m03 * m12));
        let a31 = (m00 * (m12 * m23 - m13 * m22) - m10 * (m02 * m23 - m03 * m22) + m20 * (m02 * m13 - m03 * m12));
        let a32 = -(m00 * (m11 * m23 - m13 * m21) - m10 * (m01 * m23 - m03 * m21) + m20 * (m01 * m13 - m03 * m11));
        let a33 = (m00 * (m11 * m22 - m12 * m21) - m10 * (m01 * m22 - m02 * m21) + m20 * (m01 * m12 - m02 * m11));
  
        var det = m00 * a00 + m01 * a01 + m02 * a02 + m03 * a03;
        if(det == 0)
            throw "La matriz no es invertible."

        let d = 1/det;
        
        return [ d*a00, d*a01, d*a02, d*a03,
                 d*a10, d*a11, d*a12, d*a13,
                 d*a20, d*a21, d*a22, d*a23,
                 d*a30, d*a31, d*a32, d*a33];
    }
    /**
     * M^T * V^T = (VM)^T
     * @param {*} m 
     * @param {*} v 
     */
    multiplyVector(m,v){
        let u = [];
        u[0] = v[0] * m[0] + v[1] * m[4] + v[2] * m[8] + v[3] * m[12];
        u[1] = v[0] * m[1] + v[1] * m[5] + v[2] * m[9] + v[3] * m[13];
        u[2] = v[0] * m[2] + v[1] * m[6] + v[2] * m[10] + v[3] * m[14];
        u[3] = v[0] * m[3] + v[1] * m[7] + v[2] * m[11] + v[3] * m[15];
        return u;
    } 
    transformVector(m,v){
        let u = [];
        u[0] = v[0] * m[0] + v[1] * m[4] + v[2] * m[8] + v[3] * m[12];
        u[1] = v[0] * m[1] + v[1] * m[5] + v[2] * m[9] + v[3] * m[13];
        u[2] = v[0] * m[2] + v[1] * m[6] + v[2] * m[10] + v[3] * m[14];
        return new V3(u[0], u[1], u[2]);
    }       
    degToRad(degrees){
        return degrees * (Math.PI / 180);
    }
    printT(m){
        return `{{${m[0]},${m[4]},${m[8]},${m[12]}},
                 {${m[1]},${m[5]},${m[9]},${m[13]}},
                 {${m[2]},${m[6]},${m[10]},${m[14]}},
                 {${m[3]},${m[7]},${m[11]},${m[15]}}}`;
    }
    print(m){
        return `{{${m[0]},${m[1]},${m[2]},${m[3]}},
                 {${m[4]},${m[5]},${m[6]},${m[7]}},
                 {${m[8]},${m[9]},${m[10]},${m[11]}},
                 {${m[12]},${m[13]},${m[14]},${m[15]}}}`;
    }
}

let M4 = new Matrix4();
export default M4;