"use strict";

import {resize, Select} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import PhongTexture from "../../materials/PhongTexture.js";
import {Cube, Cylinder, Sphere} from "../../geometry/Figure.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import Light from "../../geometry/Light.js";
import TexcoordGenerator from "../../materials/TexcoordGenerator.js";
import Suzanne from "../../geometry/Suzanne.js";
import Teapot from "../../geometry/Teapot.js";

/**
 * Funciones de proyección de textura.
 * MODO 0: PLANA
 * MODO 1: CILINDRICA
 * MODO 2: CUBICO (DIRECCIONAL)
 * MODO 3: ESFÉRICO
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    let srcTextures = [ "../../../images/capitulo_07/checkboard.png",
                        "../../../images/capitulo_07/textures/Bricks071_1K_Color.jpg",
                        "../../../images/capitulo_07/textures/gamrig_2k.png",
                    ];
    let textureIds = { ajedrez: 0, ladrillos: 1, paisaje: 2 };
    
    let lightPosition = [0, 1.9, .3, 0];
    let uniforms = { u_material : {}, 
                    u_light: { position: lightPosition, la: [0.3,0.3,0.3], ld: [1, 1, 1],
                                                        ls: [1.0, 1.0, 1.0]}};
    let figuresGeom = [ new Cube(1,true), new Cylinder(.5, 1, 25), new Sphere(.6, 25, 25), new Teapot()];
    
    let configMaterial = { filters: {min: gl.NEAREST, mag: gl.NEAREST, wrap_s: gl.REPEAT, wrap_t: gl.REPEAT}};
    let materials = [];
    for(var i= 0; i < figuresGeom.length; i++)
        materials.push( new TexcoordGenerator(gl, WebGL, figuresGeom[i], srcTextures, setTextures, uniforms, configMaterial)); 
    
    let scale = M4.scale(.5,.5,.5);
    let figures = [ new GeometryPainter(gl, materials[0], M4.multiply(M4.translate(-1.5,-.2,-.1), scale)),
                    new GeometryPainter(gl, materials[1], M4.multiply(M4.translate(-.5,-.2,-.1), scale)),
                    new GeometryPainter(gl, materials[2], M4.multiply(M4.translate(.5,-.2,-.1), scale)),
                    new GeometryPainter(gl, materials[3], M4.multiply(M4.translate(1.5,-.2,-.1), M4.scale(.3,.3,.3)))];

    var pos = new V3(0,0,1.5);
    let camera = new TrackballCamera(pos);
    camera.setZoomConfig(2.4, 1);
    let bulb = new Light(gl, WebGL, lightPosition);

    let textures;
    
    let mode = 3;
    let right_controls = document.getElementById("right-options");
    let proj_modes = { "plana": 0,  "cilíndrica": 1, "cúbica": 2,"esférica": 3};
    let selectUMode = new Select(right_controls, "mode", setMode, proj_modes,"esférica");
    let selectTexture = new Select(right_controls, "textures", setTextureId, textureIds, "ajedrez");
    let textureId = 0;
    
    gl.enable(gl.DEPTH_TEST);
    
    gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true);
    gl.clearColor(0, 0, 0, 0);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    
    function draw(){        
        resize(canvas);
        
        var {width, height} = gl.canvas;
        
        gl.viewport(0, 0, width, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
        
        let aspect = width/height;
        var projectionMatrix = M4.perspective(85, aspect, .1, 100);
        var viewMatrix = camera.getMatrix();
        var lightPos = M4.multiplyVector(viewMatrix, lightPosition);
        
        uniforms.u_light.position = lightPos;
        uniforms.mode = mode;
        if(textures){
            uniforms.u_material = { diffuse_map : { id: textureId, texture: textures[textureId]},
                                    shininess: 0.088*128};
        }
        for(var i = 0; i < figures.length; i++)
            figures[i].draw(gl, viewMatrix, projectionMatrix, uniforms);
        
        //bulb.draw(gl, M4.multiply(viewMatrix, M4.scale(.5, .5, .5)), projectionMatrix, lightPosition);
    }
    function setTextures(loadedTextures){
        textures = loadedTextures;
        draw();
    }
    function setMode(value){
        mode = proj_modes[value];
        draw();
    }    
    function setTextureId(value){
        textureId = textureIds[value];
        draw();
    }
} 