"use strict";

import {Slider, resize} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import VertexColor from "../../materials/VertexColor.js";
import SolidColor from "../../materials/SolidColor.js";
import { Cube, Plane, Sphere } from "../../geometry/Figure.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";

/**
 * Modelo de color RGB.
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

let rgb = [1,1,1,1];
function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    
    let cubeFigure = new Cube(1, true);
    let planeFigure = new Plane(1.65, 1.64, true);
    let sphereFigure = new Sphere(0.03, 10, 10, true);

    cubeFigure.colors = [ 1, 0, 0, 1,
                          1, 0, 1, 1,
                          1, 1, 1, 1,
                          1, 1, 0, 1,
                          0, 0, 0, 1,
                          0, 0, 1, 1,
                          0, 1, 1, 1,
                          0, 1, 0, 1];
    let initTransform =  M4.translate(-0.1,.05,0);
    let cube = new GeometryPainter(gl, new VertexColor(gl, WebGL, cubeFigure), initTransform);
    let planeUniforms = { u_color : [1,0,0,1]};
    let plane = new GeometryPainter(gl, new SolidColor(gl, WebGL, planeFigure), M4.translate(0,0,-1));
    let uniformColor = { u_color: [1,1,1,1]};
    
    let colorTest = new GeometryPainter(gl, new SolidColor(gl, WebGL, sphereFigure), initTransform);
    let baseTest = new GeometryPainter(gl, new SolidColor(gl, WebGL, sphereFigure), M4.multiply(initTransform, M4.scale(1.13,1.13,1.13)));
    let baseUniform = {u_color : [0,0,0,1]};
    
    //HTML CONTROLS
    let controls = document.getElementById("ui-container");
    controls.className = "right-box game";

    let colors = [ "#778899"];
    let sliders = [ new Slider(controls, "R", 0 ,1, updateRGB(0), rgb[0], 0.01, colors),
                    new Slider(controls, "G", 0, 1, updateRGB(1), rgb[1], 0.01, colors),
                    new Slider(controls, "B", 0, 1, updateRGB(2), rgb[2], 0.01, colors)
                  ];
    for(var i = 0; i < 3; i++)
        sliders[i].setStyle("game-slider", "game-value");
        
    var fov = 80;
    let zNear = .1;
    let zFar = 2000;
    var pos = M4.multiplyVector(M4.rotateY(30),[0,.6,1.6,1]);
    let camera = new TrackballCamera( new V3(pos[0], pos[1], pos[2]));
    camera.setZoomConfig(3,1);
    
    resize(canvas);
    
    gl.enable(gl.DEPTH_TEST);
    gl.enable(gl.CULL_FACE);
    gl.enable(gl.DEPTH_TEST);
    gl.enable(gl.SCISSOR_TEST);
    
    gl.clearColor(0, 0, 0, 0);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    
    draw();

    function draw(){
        resize(canvas);
        var {width, height} = gl.canvas;
        const smallWidth = 100;
        const smallHeight = 80;
        
        //dibuja viewport grande
        gl.scissor(0, 0, width-smallWidth, height);
        gl.viewport(0, 0, width-smallWidth, height);
        gl.clearColor(0, 0, 0, 0);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
            
        let aspect = (width-smallWidth)/height;
        var projectionMatrix = M4.perspective(fov, aspect, zNear, zFar);
        let viewMatrix = camera.getMatrix();
        
        cube.draw(gl, viewMatrix, projectionMatrix);     
        var testTransformation = M4.multiply(viewMatrix, M4.translate(-.5+rgb[2],-.5+rgb[1],-.5+rgb[0]));
        
        uniformColor.u_color = rgb;
        gl.cullFace(gl.FRONT);
        let lightness = rgbLightness(rgb[0], rgb[1], rgb[2]);
        baseUniform.u_color = [lightness, lightness, lightness, 1];
        baseTest.draw(gl, testTransformation, projectionMatrix, baseUniform);
 
        gl.cullFace(gl.BACK);
        colorTest.draw(gl, testTransformation, projectionMatrix, uniformColor);
       
        //viewport pequeño (color)
        gl.viewport(width-1.4*smallWidth, height-2.3*smallHeight, smallWidth, smallHeight);
        gl.scissor(width-1.4*smallWidth, height-2.3*smallHeight, smallWidth, smallHeight);
        gl.clearColor(0, 0, 0, 1);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
        planeUniforms.u_color = [rgb[0], rgb[1], rgb[2], 1];
        plane.draw(gl, M4.scale(aspect,1,1), projectionMatrix, planeUniforms);
    }

    function updateRGB(index){
        return function(value){
            rgb[index] = value;
            draw();
        };
    }
    function rgbLightness(r,g,b){
        let c_max = Math.max(r,g,b);
        let c_min = Math.max(r,g,b);
        let lightness = 1-((c_max + c_min)/2) + .2;
        return (lightness > 1) ? 1 : lightness;
    }
}