import {getDisconnectedVertices, getNormalsFromIndices, getNormalsFromVertices} from "../geometry/Figure.js";

/**
 * Programa de sombreado de un mapeo de textura simple con Modelo de Iluminación de Phong 
 * utilizando materiales.
 * @author Melissa Méndez Servín.
 */
export default class PhongTMaterial{
    /**
     * 
     * @param {WebGLRenderingContext} gl 
     * @param {*} WebGL 
     * @param {*} attributes 
     */
    constructor(gl, WebGL, figure, initialUniforms){
        var vsh = `#version 300 es
                    uniform mat4 u_PVM_matrix;
                    uniform mat4 u_VM_matrix;
                    uniform mat4 u_VMN_matrix;
                    
                    in vec4 a_position;
                    in vec3 a_normal;
                
                    out vec3 v_normal; 
                    out vec3 v_position; 

                    void main(){
                    
                        v_position = (u_VM_matrix * a_position).xyz;
                        v_normal = (u_VMN_matrix * vec4(a_normal, 0)).xyz;
            
                        gl_Position = u_PVM_matrix * a_position;
                    }`;
        var fsh = `#version 300 es
                    precision highp float;
                    
                    struct Material{
                        vec4 ka;
                        vec4 kd;
                        float shininess;
                    };

                    uniform  vec4 u_light_position;
                    uniform Material u_material; 

                    in vec3 v_normal;
                    in vec3 v_position;
                    
                    out vec4 glColor;
                
                    void main(){
                        vec3 L = normalize(u_light_position.xyz - v_position);
                        vec3 N = normalize(v_normal);
            
                        float cos_angle = max(dot(N, L), 0.0);
                        vec4 contribution = u_material.ka * vec4(vec3(.2),1.) + u_material.kd * cos_angle;

                        glColor = contribution;
                    }`; 

        if (WebGL.programs["PhongTM"])
            this.program = WebGL.programs["PhongTM"];
        else
            this.program = WebGL.createProgram(gl, vsh, fsh, "PhongTM");
            
        this.vertices = (figure.byIndices || figure.getNormals) ?  figure.getVertices() : getDisconnectedVertices(figure.getVertices(), figure.getFaces());
        if(figure.getNormals)
            this.normals = figure.getNormals();
        else
            this.normals = (figure.byIndices) ? getNormalsFromIndices(figure.getFaces(), this.vertices) : getNormalsFromVertices(this.vertices);
        
        let attributes = {  position: { numComponents: 3, 
                                                 data: this.vertices},
                              normal: { numComponents: 3, 
                                                 data: this.normals}
                         };
        
        if(figure.byIndices){
            let vaoAndIndices = WebGL.setVAOAndAttributes(gl, this.program, attributes, figure.getFaces());
            this.vao = vaoAndIndices.vao;
            this.indexBuffer = vaoAndIndices.indexBuffer;
            this.numIndices = figure.numIndices;
        }else{
            this.vao = WebGL.setVAOAndAttributes(gl, this.program, attributes);  
            this.numElements = this.vertices.length/3;
        }
        this.setUniforms = WebGL.setUniforms(gl, this.program);
        this.uniforms = Object.assign({}, initialUniforms);
    }
}