import {getDisconnectedVertices, getNormalsFromIndices, getNormalsFromVertices} from "../geometry/Figure.js";

/**
 * Programa de sombreado del Modelo de Iluminación difuso.
 * @author Melissa Méndez Servín.
 */
export default class Diffuse {
  
    constructor(gl, WebGL, figure, initialUniforms){
        var vsh = `#version 300 es
                    uniform mat4 u_PVM_matrix;
                    uniform mat4 u_VM_matrix;
                    uniform mat4 u_VMN_matrix;
                    
                    in vec4 a_position;
                    in vec3 a_normal;
                
                    out vec3 v_normal; 
                    out vec3 v_position; 

                    void main(){
                    
                        v_position = (u_VM_matrix * a_position).xyz;
                        v_normal = (u_VMN_matrix * vec4(a_normal, 0)).xyz;
            
                        gl_Position = u_PVM_matrix * a_position;
                    }`;
        var fsh = `#version 300 es
                    precision highp float;
                    
                    uniform vec4 u_light_position;
                    uniform vec4 u_color;
                    
                    in vec3 v_normal;
                    in vec3 v_position;
                    
                    out vec4 glColor;
                    
                    vec3 ambient = vec3(0.25, 0.25, 0.25);
                
                    void main(){
                        vec3 to_light = normalize( u_light_position.xyz - v_position );
                        vec3 fragment_normal = normalize(v_normal);
            
                        float cos_angle = max(dot(fragment_normal, to_light), 0.0);
            
                        glColor = vec4(u_color.rgb * cos_angle + ambient * u_color.rgb, u_color.a);
                    }`; 

        if (WebGL.programs["Diffuse"])
            this.program = WebGL.programs["Diffuse"];
        else
            this.program = WebGL.createProgram(gl, vsh, fsh, "Diffuse");
            
        this.vertices = (figure.byIndices || figure.getNormals) ?  figure.getVertices() : getDisconnectedVertices(figure.getVertices(), figure.getFaces());
        if(figure.getNormals)
            this.normals = figure.getNormals();
        else
            this.normals = (figure.byIndices) ? getNormalsFromIndices(figure.getFaces(), this.vertices) : getNormalsFromVertices(this.vertices);
        
        let attributes = {  position: { numComponents: 3, 
                                                 data: this.vertices},
                              normal: { numComponents: 3, 
                                                 data: this.normals}
                         };
        
        if(figure.byIndices){
            let vaoAndIndices = WebGL.setVAOAndAttributes(gl, this.program, attributes, figure.getFaces());
            this.vao = vaoAndIndices.vao;
            this.indexBuffer = vaoAndIndices.indexBuffer;
            this.numIndices = figure.numIndices;
        }else{
            this.vao = WebGL.setVAOAndAttributes(gl, this.program, attributes);  
            this.numElements = this.vertices.length/3;
        }
        this.setUniforms = WebGL.setUniforms(gl, this.program);
        this.uniforms = Object.assign({}, initialUniforms);
    }
}