"use strict";

import {resize, Slider, addLabel, resizeAndCenterX, DarkMode, Select} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import MaterialMapping from "../../materials/MaterialMapping.js";
import {Cube, Sphere} from "../../geometry/Figure.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import Light from "../../geometry/Light.js";

/**
 * Material Mapping.
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    let cubeFigure = new Cube();
    let sphereFigure = new Sphere(.85,200, 200, false);
    let srcTextures = ["../../../images/capitulo_07/textures/Container_Color.png",
                      "../../../images/capitulo_07/textures/Container_Specular.png",
                      "../../../images/capitulo_07/textures/PaintedMetal004_1K_Color.jpg",
                      "../../../images/capitulo_07/textures/PaintedMetal004_1K_Metalness.jpg"];
    
    let lightPosition = [0, 0.8, 2.5, 0];
    let uniforms = { u_material : {}, 
                     u_light: { position: lightPosition, la: [0.1,0.1,0.1], ld: [1, 1, 1],
                                ls: [1.0, 1.0, 1.0]}};

    let figures = [ new GeometryPainter(gl, new MaterialMapping(gl, WebGL, cubeFigure, srcTextures, setTextures, uniforms), M4.translate(0,-.1,0)),
                    new GeometryPainter(gl, new MaterialMapping(gl, WebGL, sphereFigure, srcTextures, setTextures, uniforms), M4.translate(0,-.1,-.3))];

    var pos = new V3(0,0,1.5);
    let camera = new TrackballCamera(pos);
    camera.setZoomConfig(2, 1);
    let bulb = new Light(gl, WebGL, lightPosition, 0.05);

    let textures;
    var viewMatrix;
    
    let light_controls = document.getElementById("ui-container-up");
    addLabel(light_controls, "Posición de la luz");
    var light_pos_x = new Slider(light_controls, "x", -10, 10, upadeteLPos(0), lightPosition[0], 0.01, ["#4994D0"]);
    var light_pos_y = new Slider(light_controls, "y", -10, 10, upadeteLPos(1), lightPosition[1], 0.01, ["#4994D0"]);
    var light_pos_z = new Slider(light_controls, "z", -10, 10, upadeteLPos(2), lightPosition[2], 0.01, ["#4994D0"]);

    let figuresIds = { caja: 0, esfera: 2};
    var figureId = 0;
    let right_controls = document.getElementById("right-options");
    let darkMode = new DarkMode(draw, [light_controls]);
    let selectFigure = new Select(right_controls, "figuras", setFigureId, figuresIds, "caja");
 
    gl.enable(gl.DEPTH_TEST);
    
    gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true);
    gl.clearColor(0, 0, 0, 0);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    
    function draw(){        
        resize(canvas);
        resizeAndCenterX(light_controls, {});
        darkMode.check(gl);
        
        var {width, height} = gl.canvas;
        
        gl.viewport(0, 0, width, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
        
        let aspect = width/height;
        var projectionMatrix = M4.perspective(90, aspect, .1, 100);
        viewMatrix = camera.getMatrix();
        var lightPos = M4.multiplyVector(viewMatrix, lightPosition);
        
        uniforms.u_light.position = lightPos;
        if(textures){
            uniforms.u_material = { diffuse_map : { id: figureId, texture: textures[figureId]},
                                    specular_map : { id: figureId+1, texture: textures[figureId+1]},
                                    shininess: 0.088*128};
            
            figures[figureId/2].draw(gl, viewMatrix, projectionMatrix, uniforms);
        }
        bulb.draw(gl, viewMatrix, projectionMatrix, lightPosition);
    }
    function setTextures(loadedTextures){
        textures = loadedTextures;
        draw();
    }
    function upadeteLPos(index){
        return function(value){
            lightPosition[index] = value;
            draw();
        }
    }
    function setFigureId(value){
        figureId = figuresIds[value];
        draw();
    }
} 