"use strict";

import {resize, Slider, Color, addLabel, DarkMode, TextBox, Select, resizeAndCenterX} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import Phong from "../../materials/Phong.js";
import {Cube, Sphere} from "../../geometry/Figure.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";
import Teapot from "../../geometry/Teapot.js";
import Bunny from "../../geometry/Bunny.js";
import Light from "../../geometry/Light.js";

/**
 * Modelo de reflexión difusa.
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    
    let sphereFigure = new Sphere(1, 30,30);
    let cubeFigure = new Cube(1.5);
    let teapotFigure = new Teapot();
    let bunnyFigure = new Bunny();
    
    let lightPosition = [2.3, 1.11, -.34, 0];
    let material = { ka: 0, kd: 1, ks: 0};
    let light = { position: lightPosition, ka: .1, la: [0,0,0], kd: .9, ld: [1, 1, 1],
                  ls: [1.0, 1.0, 1.0], shininess: 0};
    let diffuseUniforms = { u_light: light, u_material: material}; 
    let figures = {
        tetera: new GeometryPainter(gl, new Phong(gl, WebGL, teapotFigure, diffuseUniforms)),
        esfera : new GeometryPainter(gl, new Phong(gl, WebGL, sphereFigure, diffuseUniforms)),
        conejo: new GeometryPainter(gl, new Phong(gl, WebGL, bunnyFigure, diffuseUniforms)),
        cubo: new GeometryPainter(gl, new Phong(gl, WebGL, cubeFigure, diffuseUniforms))};
    let selectedFigure = figures.tetera;  
    
    var fov = 98;

    let zNear = .1;
    let zFar = 100;
    var pos = new V3(0,1.3,2.3);
    let camera = new TrackballCamera(pos);
    
    let controls = document.getElementById("ui-container");
    controls.className = "big-bottom-box";
    var kd_slider = new Slider(controls, "k<sub>D</sub>", 0, 1, upadeteKD, material.kd, 0.01, ["#4994D0"]);
    var diffuse_color = new Color(controls, "L<sub>D</sub>", setColor, "#FFFFFF");
    let light_controls = document.getElementById("ui-container-up");
    addLabel(light_controls, "Posición de la luz");
    var light_pos_x = new Slider(light_controls, "x", -10, 10, upadeteLPos(0), lightPosition[0], 0.01, ["#4994D0"]);
    var light_pos_y = new Slider(light_controls, "y", -10, 10, upadeteLPos(1), lightPosition[1], 0.01, ["#4994D0"]);
    var light_pos_z = new Slider(light_controls, "z", -10, 10, upadeteLPos(2), lightPosition[2], 0.01, ["#4994D0"]);
    
    let all_controls = [controls, light_controls];
    let darkMode = new DarkMode(draw, all_controls);
  
    let ccLabel = new TextBox(container, { left: 20, top: 55, type: "px"}, "", "author-small");

    /* Selects */
    let right_controls = document.getElementById("right-options");
    let selectFigure = new Select(right_controls, "figures", setFigure, figures);
    //Luz
    let bulb = new Light(gl, WebGL, lightPosition);

    gl.enable(gl.DEPTH_TEST);
    gl.enable(gl.CULL_FACE);
    
    gl.clearColor(0, 0, 0, 0);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    
    draw();

    function draw(){        
        resize(canvas);
        resizeAndCenterX(light_controls, {});
        var {width, height} = gl.canvas;
        
        darkMode.check(gl);

        gl.viewport(0, 0, width, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
             
        let aspect = width/height;
        var projectionMatrix = M4.perspective(fov, aspect, zNear, zFar);

        let viewMatrix = camera.getMatrix();
        var lightPos = M4.multiplyVector(viewMatrix, lightPosition);
        
        diffuseUniforms = { u_light: light, u_material: material}; 
        diffuseUniforms.u_light.position = lightPos;
        
        bulb.draw(gl, viewMatrix, projectionMatrix, lightPosition, [light.ld[0], light.ld[1], light.ld[2], 1]);
        selectedFigure.draw(gl, viewMatrix, projectionMatrix, diffuseUniforms);       
    }
    function setFigure(value){
        if(value == 'conejo'){
            ccLabel.setText( "<i>Bunny</i>. Modelo obtenido de <i>Stanford University <br>Computer Graphics Laboratory</i>.");
        }else
            ccLabel.setText("");
        selectedFigure = figures[value];
        draw();
    }
    function upadeteKD(value){
        material.kd = value;
        draw();
    }
    function setColor(value){
        light.ld = value;
        draw();
    }
    function upadeteLPos(index){
        return function(value){
            lightPosition[index] = value;
            draw();
        }
    }
}